<?php
/**
 * Elementor Widget
 * @package Travelor
 * @since 1.0.0
 */

namespace Elementor;

class Travelor_Tour_Grid_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'travelor-tour-grid-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Tour Grid', 'travelor-core');
    }

    public function get_keywords()
    {
        return ['grid', 'Tour', 'Travelor'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-slider-push';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['travelor_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'destination_settings_section',
            [
                'label' => esc_html__('Tour Settings', 'travelor-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control('tour_grid', [
            'label' => esc_html__('Tour Grid', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'col-xl-2' => esc_html__('col-xl-2', 'travelor-core'),
                'col-xl-3' => esc_html__('col-xl-3', 'travelor-core'),
                'col-xl-4' => esc_html__('col-xl-4', 'travelor-core'),
                'col-xl-6' => esc_html__('col-xl-6', 'travelor-core'),
                'col-xl-12' => esc_html__('col-xl-12', 'travelor-core'),
            ),
            'default' => 'col-xl-4',
            'description' => esc_html__('Select Tour Grid', 'travelor-core')
        ]);
        $this->add_control(
            'button_text',
            [
                'label' => __( 'Button Text', 'travelor-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => __( 'Explore More', 'travelor-core' ),
                'label_block' => true,
                'default' => __( 'Wxplore More', 'travelor-core' ),
            ]
        );
        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'travelor-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('Enter how many course you want in masonry , enter -1 for unlimited course.')
        ]);
        $this->add_control('category', [
            'label' => esc_html__('Category', 'travelor-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => travelor()->get_terms_names('tour-cat', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'travelor-core'),
        ]);
        $this->add_control('order', [
            'label' => esc_html__('Order', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'travelor-core'),
                'DESC' => esc_html__('Descending', 'travelor-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'travelor-core')
        ]);
        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'travelor-core'),
                'title' => esc_html__('Title', 'travelor-core'),
                'date' => esc_html__('Date', 'travelor-core'),
                'rand' => esc_html__('Random', 'travelor-core'),
                'comment_count' => esc_html__('Most Comments', 'travelor-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'travelor-core')
        ]);

        $this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} h4.project-two-title" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_control('title_hover_color', [
            'label' => esc_html__('Title Hover Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} h4.project-two-title .hover-text-secondary:hover" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'travelor-core'),
            'name' => 'title_typography',
            'description' => esc_html__('title typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} h4.project-two-title"
        ]);

        $this->add_control('destination_color', [
            'label' => esc_html__('Destination Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .project-two-paragraph" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Destination Typography', 'travelor-core'),
            'name' => 'dest_typography',
            'description' => esc_html__('Destination typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} .project-two-paragraph"
        ]);

        
        $this->end_controls_section();

        $this->start_controls_section(
            'styling_section',
            [
                'label' => esc_html__('Styling Settings', 'travelor-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control('subtitle_color', [
            'label' => esc_html__('Subtitle Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .section-subtitle" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Subtitle Typography', 'travelor-core'),
            'name' => 'subtitle_typography',
            'description' => esc_html__('Subtitle typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} .section-subtitle.font-sofia.tw-text-2xl"
        ]);
        $this->add_control('title2_color', [
            'label' => esc_html__('Title Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} h2.section-title" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [ 
            'label' => esc_html__('Title Typography', 'travelor-core'),
            'name' => 'title2_typography',
            'description' => esc_html__('Title typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} h2.section-title"
        ]);
        $this->add_control('description_color', [
            'label' => esc_html__('Description Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} p.tw-text-lg" => "color: {{VALUE}} !important"
            ],
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [ 
            'label' => esc_html__('Description Typography', 'travelor-core'),
            'name' => 'desc_typography',
            'description' => esc_html__('Description typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} p.tw-text-lg"
        ]);
                $this->add_responsive_control(
			'title_margin',
			[
				'label' => __( 'Title Margin', 'travelor-core' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					"{{WRAPPER}} .tw-mb-14" => "margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important",
				],
			]
		);
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        //setup query
        $args = array(
            'post_type' => 'tour',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'tour-cat',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
        ?>

            <div class="package-ip-area">
                <div class="container">
                    <div class="row">
                        <?php
                        while ($post_data->have_posts()) : $post_data->the_post();
                            $post_id = get_the_ID();
                            $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                            $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
                            $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                            $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                            $tour_single_meta_data = get_post_meta(get_the_ID(), 'travelor_tour_options', true);
                            
                            $social_icons = isset($tour_single_meta_data['social-icons']) && !empty($tour_single_meta_data['social-icons']) ? $tour_single_meta_data['social-icons'] : '';

                            // icon
                            $tour_icon = isset($tour_single_meta_data['tour_icon']) && !empty($tour_single_meta_data['tour_icon']) ? $tour_single_meta_data['tour_icon'] : '';

                            ?>
                            <div class="<?php echo esc_attr($settings['tour_grid']); ?> col-lg-6 col-md-6">
                                 <div class="service-two-wrapper bg-white tw-p-4 tw-rounded-xl tw-mb-8" data-aos="fade-up" data-aos-duration="1000" data-aos-delay="200">
                                    <div class="service-two-thumb tw-mb-5">
                                        <a href="<?php the_permalink(); ?>">
                                            <img class="tw-rounded-xl w-100" src="<?php echo esc_url($img_url) ?>" alt="thumb"></a>
                                    </div>
                                    <div class="service-two-content tw-px-2 tw-mb-2">
                                        <span class="service-two-location">
                                            <i class="ph ph-map-pin"></i>
                                            <?php echo !empty($tour_single_meta_data['tour_location']) ? esc_html($tour_single_meta_data['tour_location']) : ''; ?>
                                        </span>
                                        <h4 class="tw-text-8 fw-normal text-capitalize tw-mb-2">
                                            <a class="hover-text-secondary" href="<?php the_permalink(); ?>">
                                                <?php echo esc_html(get_the_title($post_id)) ?>
                                            </a>
                                        </h4>
                                        <p class="service-two-paragraph tw-mb-5">
                                            <?php echo !empty($tour_single_meta_data['tour_content']) ? esc_html($tour_single_meta_data['tour_content']) : ''; ?>
                                        </p>
                                        <div class="service-two-wrap tw-rounded-xl tw-py-4 tw-px-6">
                                            <div class="service-two-star d-flex tw-gap-6 tw-pb-4 tw-mb-6">
                                                <span class="text-main-600 fw-medium d-flex tw-gap-2">
                                                    <i class="ph ph-star"></i> 
                                                    <?php echo !empty($tour_single_meta_data['tour_rating']) ? esc_html($tour_single_meta_data['tour_rating']) : ''; ?>
                                                </span>
                                                <span class="text-main-600 fw-medium">
                                                    <?php echo !empty($tour_single_meta_data['tour_day']) ? esc_html($tour_single_meta_data['tour_day']) : ''; ?>
                                                </span>
                                            </div>
                                            <div class="d-flex align-items-center justify-content-between flex-wrap row-gap-3">
                                                <div class="service-two-price">
                                                    <h6>
                                                        <?php echo !empty($tour_single_meta_data['tour_price']) ? esc_html($tour_single_meta_data['tour_price']) : ''; ?>
                                                    </h6>
                                                    <p>
                                                        <?php echo !empty($tour_single_meta_data['tour_perperson']) ? esc_html($tour_single_meta_data['tour_perperson']) : ''; ?>
                                                    </p>
                                                </div>
                                                <div>
                                                    <a class="font-heading tw-text-sm text-uppercase text-main-600 fw-bold hover-text-secondary" href="<?php the_permalink(); ?>">
                                                        <?php echo $settings['button_text']; ?>
                                                        <i class="tw-text-base ph ph-arrow-up-right"></i>
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    </div>
                </div>
            </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Travelor_Tour_Grid_Widget());