<?php
/**
 * Elementor Widget
 * @package travelor
 * @since 1.0.0
 */

namespace Elementor;

class Travelor_Team_Grid_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'travelor-team-grid-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Team grid', 'travelor-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['travelor_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'team_settings_section',
            [
                'label' => esc_html__('Team Settings', 'travelor-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control('team_grid', [
            'label' => esc_html__('Team Grid', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'col-xl-2' => esc_html__('col-xl-2', 'travelor-core'),
                'col-xl-3' => esc_html__('col-xl-3', 'travelor-core'),
                'col-xl-4' => esc_html__('col-xl-4', 'travelor-core'),
                'col-xl-6' => esc_html__('col-xl-6', 'travelor-core'),
                'col-xl-12' => esc_html__('col-xl-12', 'travelor-core'),
            ),
            'default' => 'col-xl-4',
            'description' => esc_html__('Select Team Grid', 'travelor-core')
        ]);
        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'travelor-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many course you want in masonry , enter -1 for unlimited course.')
        ]);
        $this->add_control('category', [
            'label' => esc_html__('Category', 'travelor-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => travelor()->get_terms_names('team-cat', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'travelor-core'),
        ]);
        $this->add_control('order', [
            'label' => esc_html__('Order', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'travelor-core'),
                'DESC' => esc_html__('Descending', 'travelor-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'travelor-core')
        ]);
        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'travelor-core'),
                'title' => esc_html__('Title', 'travelor-core'),
                'date' => esc_html__('Date', 'travelor-core'),
                'rand' => esc_html__('Random', 'travelor-core'),
                'comment_count' => esc_html__('Most Comments', 'travelor-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'travelor-core')
        ]);

        $this->add_control('name_color', [
            'label' => esc_html__('Name Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} h4" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('name Typography', 'travelor-core'),
            'name' => 'name_typography',
            'description' => esc_html__('name typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} .tw-text-705"
        ]);

        
       $this->add_responsive_control(
			'team_padding',
			[
				'label' => __( 'Team Padding', 'travelor-core' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					"{{WRAPPER}} .team-three-area.pb-140" => "padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important",
				],
			]
		);
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        //setup query
        $args = array(
            'post_type' => 'team',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'team-cat',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
        ?>

            <div class="team-three-area pb-140 position-relative z-1">
                <div class="container">
                    <div class="row">
                        <?php
                        while ($post_data->have_posts()) : $post_data->the_post();
                            $post_id = get_the_ID();
                            $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                            $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
                            $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                            $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                            $team_single_meta_data = get_post_meta(get_the_ID(), 'travelor_team_options', true);

                            
                            $social_icons = isset($team_single_meta_data['social-icons']) && !empty($team_single_meta_data['social-icons']) ? $team_single_meta_data['social-icons'] : '';

                            // icon
                            $team_icon = isset($team_single_meta_data['team_icon']) && !empty($team_single_meta_data['team_icon']) ? $team_single_meta_data['team_icon'] : '';

                            ?>

                            <div class="col-lg-6 col-md-6 <?php echo esc_attr($settings['team_grid']); ?>">
                                <div class="team-three-wrapper tw-mb-8 overflow-hidden position-relative z-1" data-aos="fade-up" data-aos-duration="1000" data-aos-delay="200">
                                    <div>
                                        <img class="w-100" src="<?php echo esc_url($img_url) ?>" alt="">
                                    </div>
                                    <div class="team-three-content position-absolute text-center">
                                        <h4 class="tw-text-705 fw-normal tw-mb-3">
                                            <a href="<?php the_permalink(); ?>">
                                                <?php echo esc_html(get_the_title($post_id)) ?>
                                            </a>
                                        </h4>
                                        <div class="team-three-social">
                                            <ul class="d-flex justify-content-center tw-gap-4">
                                                 <?php
                                                    if (!empty($social_icons)) {
                                                        foreach ($social_icons as $item) {
                                                            printf('<li><a class="text-main-600 tw-text-xl" href="%1$s">
                                                            <i class="%2$s"></i></a></li>', $item['url'], $item['icon']);
                                                        }
                                                    }
                                                ?> 
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    </div>
                </div>
            </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Travelor_Team_Grid_Widget());