<?php
/**
 * Elementor Widget
 * @package Travelor
 * @since 1.0.0
 */

namespace Elementor;

class Travelor_Destination_Slider_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'travelor-destination-slider-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Destination Slider', 'travelor-core');
    }

    public function get_keywords()
    {
        return ['Tour', 'Destination', 'Travelor'];
    }

    /**
     * Get widget icon.
     * 
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-slider-push';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['travelor_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'destination_settings_section',
            [
                'label' => esc_html__('Destination Settings', 'travelor-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'show_bg_image',
            [
                'label' => __( 'Show / Hide BG Img', 'travelor-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'Show', 'travelor-core' ),
                'label_off' => __( 'Hide', 'travelor-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'destination_img', [
                'label' => esc_html__('Destination BG image', 'travelor-core'),
                'type' => Controls_Manager::MEDIA,
                'show_label' => false,
                'description' => esc_html__('Destination Img', 'travelor-core'),
                
            ]
        );
        $this->add_control(
            'show_shape',
            [
                'label' => __( 'Show / Hide Shape', 'travelor-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'Show', 'travelor-core' ),
                'label_off' => __( 'Hide', 'travelor-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'destination_shape', [
                'label' => esc_html__('Destination Shape', 'travelor-core'),
                'type' => Controls_Manager::MEDIA,
                'show_label' => false,
                'description' => esc_html__('Destination Shape', 'travelor-core'),
                
            ]
        );
        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'travelor-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('Enter how many course you want in masonry , enter -1 for unlimited course.')
        ]);
        $this->add_control('category', [
            'label' => esc_html__('Category', 'travelor-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => travelor()->get_terms_names('destination-cat', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'travelor-core'),
        ]);
        $this->add_control('order', [
            'label' => esc_html__('Order', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'travelor-core'),
                'DESC' => esc_html__('Descending', 'travelor-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'travelor-core')
        ]);
        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'travelor-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'travelor-core'),
                'title' => esc_html__('Title', 'travelor-core'),
                'date' => esc_html__('Date', 'travelor-core'),
                'rand' => esc_html__('Random', 'travelor-core'),
                'comment_count' => esc_html__('Most Comments', 'travelor-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'travelor-core')
        ]);

        $this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} h4.project-two-title" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_control('title_hover_color', [
            'label' => esc_html__('Title Hover Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} h4.project-two-title .hover-text-secondary:hover" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'travelor-core'),
            'name' => 'title_typography',
            'description' => esc_html__('title typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} h4.project-two-title"
        ]);

        $this->add_control('destination_color', [
            'label' => esc_html__('Destination Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .project-two-paragraph" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Destination Typography', 'travelor-core'),
            'name' => 'dest_typography',
            'description' => esc_html__('Destination typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} .project-two-paragraph"
        ]);

        $this->add_control('listing_color', [
            'label' => esc_html__('Listing Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .text-main-600" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_control('listing_bg_color', [
            'label' => esc_html__('Listing BG Color', 'travelor-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .bg-main-two-600" => "color: {{VALUE}} !important"
            ]
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Listing Typography', 'travelor-core'),
            'name' => 'listing_typography',
            'description' => esc_html__('Listing typography', 'travelor-core'),
            'selector' => "{{WRAPPER}} .text-main-600"
        ]);
        
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        //setup query
        $args = array(
            'post_type' => 'destination',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'destination-cat',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
		
		echo '
            <script>
                jQuery(document).ready(function($) {
					  var slider = new Swiper(".project-two-active", {
						slidesPerView: 4,
						spaceBetween: 30,
						loop: true,
						speed: 3000,
						autoplay:true,
						centeredSlides: true,
						breakpoints: {
						  "1800": {
							slidesPerView: 3.6,
						  },
						  "1700": {
							slidesPerView: 3.4,
						  },
						  "1400": {
							slidesPerView: 2.8,
						  },
						  "1200": {
							slidesPerView: 2.4,
						  },
						  "992": {
							slidesPerView: 2,
						  },
						  "768": {
							slidesPerView: 1.7,
						  },
						  "576": {
							slidesPerView: 1.5,
						  },
						  "0": {
							slidesPerView: 1,
						  },
						},
					  });
					});
            </script>';
        ?>

            <div class="destination position-relative z-1 overflow-hidden">
                <div class="project-two-slider">
                    <div class="project-two-active swiper-container">
                        <div class="swiper-wrapper">
                            <?php
                            while ($post_data->have_posts()) : $post_data->the_post();
                                $post_id = get_the_ID();
                                $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                                $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
                                $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                                $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                                $destination_single_meta_data = get_post_meta(get_the_ID(), 'travelor_destination_options', true);
                                
                                $social_icons = isset($destination_single_meta_data['social-icons']) && !empty($destination_single_meta_data['social-icons']) ? $destination_single_meta_data['social-icons'] : '';

                                // icon
                                $destination_icon = isset($destination_single_meta_data['destination_icon']) && !empty($destination_single_meta_data['destination_icon']) ? $destination_single_meta_data['destination_icon'] : '';

                                ?>

                                <div class="project-two-wrapper d-flex align-items-center tw-gap-6 bg-white tw-rounded-xl tw-py-11 tw-px-11 position-relative z-1 overflow-hidden flex-wrap row-gap-5 swiper-slide">
                                    <div class="project-two-thumb">
                                        <a href="<?php the_permalink(); ?>">
                                            <img src="<?php echo esc_url($img_url) ?>" alt="">
                                        </a>
                                    </div>
                                    <div>
                                        <p class="project-two-paragraph fw-medium mb-0">
                                            <?php echo $destination_single_meta_data['destination_subtitle'] ?>
                                        </p>
                                        <h4 class="project-two-title tw-text-10 fw-normal">
                                            <a class="hover-text-secondary" href="<?php the_permalink(); ?>">
                                                <?php echo esc_html(get_the_title($post_id)) ?>
                                            </a>
                                        </h4>
                                    </div>
                                    <div class="project-two-text bg-main-two-600 text-main-600 position-absolute">
                                        <?php echo $destination_single_meta_data['destination_listing'] ?>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        </div>
                    </div>    
                </div>
                <?php if( 'yes'===$settings['show_bg_image'] ){ ?>
                    <?php if($settings['destination_img']['url']) : ?>
                        <div class="project-bg">
                            <img src="<?php echo $settings['destination_img']['url']; ?>" alt="">
                        </div>
                    <?php endif; ?>
                <?php } ?>
                <?php if( 'yes'===$settings['show_shape'] ){ ?>
                    <?php if($settings['destination_shape']['url']) : ?>
                        <div class="project-bg-shape">
                            <img class="project-bg-shape-3 position-absolute start-0 z-1" src="<?php echo $settings['destination_shape']['url']; ?>" alt="">  
                        </div>
                    <?php endif; ?>
                <?php } ?>
            </div>

        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Travelor_Destination_Slider_Widget());